import numpy as np
import pandas as pd
import matplotlib.pyplot as plt
from types import SimpleNamespace


def initialize():
    """
    cfg: 設定変数をまとめたSimpleNamespaceを返す
    - infile: 入力Excelファイル名
    - p: 多項式次数
    - sigma_noise: ノイズ標準偏差の初期値
    - m0: 事前分布の平均ベクトル
    - sigma0: 事前分布の共分散スカラー
    - em_iters: エビデンス最適化の反復回数
    - em_tol: 収束判定の許容誤差
    """
    cfg = SimpleNamespace()
    cfg.infile = "random-poly.xlsx"
    cfg.p = 5
    cfg.sigma_noise = 1.0         # ノイズ標準偏差 σ_noise
    cfg.m0 = np.zeros(cfg.p + 1)  # 事前平均ベクトル
    cfg.sigma0 = 1.0              # 事前共分散スカラー（共分散行列 = sigma0 * I）
    cfg.em_iters = 50
    cfg.em_tol = 1e-6
    return cfg


def read_data(infile):
    """
    Excelファイルを読み込む。
    1行目をラベル、2行目以降をデータとする。
    returns: labels (list), x_data, y_data
    """
    df = pd.read_excel(infile, header=0)
    labels = df.columns.tolist()
    x = df.iloc[:, 0].values
    y = df.iloc[:, 1].values
    return labels, x, y


def evidence_maximization(x, y, cfg):
    """
    エビデンス最大化で m, S, sigma_noise および事後パラメータを返す
    returns: m_N, S_N, sigma_noise, alpha, beta
    """
    N = x.shape[0]
    p = cfg.p
    # デザイン行列 Φ
    Phi = np.vstack([x**i for i in range(p+1)]).T  # (N, p+1)
    M = p + 1
    I = np.eye(M)

    # 初期化
    sigma_noise = cfg.sigma_noise
    beta = 1.0 / (sigma_noise**2)                # ノイズ精度
    alpha = 1.0 / cfg.sigma0                    # 事前精度
    m = cfg.m0.copy()                           # 初期事前平均

    # 固有値はループ外で一度計算
    eigvals = np.linalg.eigvalsh(Phi.T @ Phi)

    for _ in range(cfg.em_iters):
        # 事後分布の計算
        S_inv = alpha * I + beta * (Phi.T @ Phi)
        S = np.linalg.inv(S_inv)
        m = beta * (S @ (Phi.T @ y)) + alpha * (S @ cfg.m0)

        # 有効自由度 γ の計算
        gamma = np.sum(beta * eigvals / (alpha + beta * eigvals))

        # ハイパーパラメータ更新
        alpha_new = gamma / ((m - cfg.m0) @ (m - cfg.m0))
        residual = y - (Phi @ m)
        beta_new = (N - gamma) / (residual @ residual)

        if abs(alpha_new - alpha) < cfg.em_tol * alpha and abs(beta_new - beta) < cfg.em_tol * beta:
            alpha, beta = alpha_new, beta_new
            break
        alpha, beta = alpha_new, beta_new

    # 最終的な事後
    S_inv = alpha * I + beta * (Phi.T @ Phi)
    S_N = np.linalg.inv(S_inv)
    m_N = beta * (S_N @ (Phi.T @ y)) + alpha * (S_N @ cfg.m0)
    sigma_noise = np.sqrt(1.0 / beta)

    return m_N, S_N, sigma_noise, alpha, beta


def plot_results(x, y, m, S, sigma_noise, p):
    """
    データと回帰結果（平均、±1σ（モデル）、±1σ（モデル+ノイズ））を描画
    """
    x_plot = np.linspace(x.min(), x.max(), 300)
    Phi_plot = np.vstack([x_plot**i for i in range(p+1)]).T

    y_mean = Phi_plot @ m
    var_mean = np.sum((Phi_plot @ S) * Phi_plot, axis=1)
    sigma_model = np.sqrt(var_mean)
    sigma_full = np.sqrt(var_mean + sigma_noise**2)

    plt.figure(figsize=(8, 6))
    plt.scatter(x, y, c='k', label='Data')
    plt.plot(x_plot, y_mean, lw=2, label='Predictive mean')
    plt.fill_between(x_plot, y_mean - sigma_model, y_mean + sigma_model,
                     alpha=0.3, label='±1σ (model)')
    plt.fill_between(x_plot, y_mean - sigma_full, y_mean + sigma_full,
                     alpha=0.2, label='±1σ (model+noise)')
    plt.xlabel('x')
    plt.ylabel('y')
    plt.title(f'Bayesian poly regression (p={p}), σ_noise={sigma_noise:.4f}')
    plt.legend()
    plt.show()


def main():
    cfg = initialize()
    labels, x_data, y_data = read_data(cfg.infile)
    print("Labels:", labels)

    m_N, S_N, sigma_noise_est, alpha_est, beta_est = evidence_maximization(x_data, y_data, cfg)

    print(f"Estimated sigma_noise = {sigma_noise_est:.4f}, α = {alpha_est:.4f}, β = {beta_est:.4f}")
    means = m_N
    stds = np.sqrt(np.diag(S_N))
    for i, (mu, sigma) in enumerate(zip(means, stds)):
        print(f"w_{i}: mean = {mu:.4f}, std = {sigma:.4f}")

    plot_results(x_data, y_data, m_N, S_N, sigma_noise_est, cfg.p)


if __name__ == "__main__":
    main()
