import sys
import csv
import openpyxl
import numpy as np
import matplotlib.pyplot as plt
from types import SimpleNamespace

def initialize():
    cfg = SimpleNamespace()
    cfg.infile = None
    cfg.xlabel = 'x'
    cfg.ylabel = 'y'
    cfg.p = 5
    cfg.n = 100
    cfg.mu0 = np.zeros(cfg.p)
    cfg.sigma0 = np.eye(cfg.p)
    cfg.sigma_noise = 1.0  # 初期ノイズ推定値
    cfg.tol = 1e-4         # ノイズ収束判定閾値
    cfg.max_iter = 100     # ノイズ収束最大繰り返し数
    return cfg

def update_vars(cfg):
    #argvからkey=valを読み込んでcfgに代入
    for arg in sys.argv[1:]:
        if '=' in arg:
            key, value = arg.split('=')
            if hasattr(cfg, key):
                try:
                    # cfgの属性を適切な型に変換
                    if isinstance(getattr(cfg, key), bool):
                        setattr(cfg, key, value.lower() == 'true')
                    elif isinstance(getattr(cfg, key), int):
                        setattr(cfg, key, int(value))
                    elif isinstance(getattr(cfg, key), float):
                        setattr(cfg, key, float(value))
                    else:
                        setattr(cfg, key, value)
                except ValueError:
                    print(f"Invalid value for {key}: {value}")
            else:
                print(f"Unknown parameter: {key}")

def basis(i, x):
    if i == 0: return 1.0
    return x ** i

def model(x, ai, cfg):
    return sum(ai[i] * basis(i, x) for i in range(cfg.p))

def design_matrix(x_data, cfg):
    return np.array([[basis(i, x) for i in range(cfg.p)] for x in x_data])

def read_data(infile):
    # ファイルの拡張子を取得
    ext = infile.split('.')[-1].lower()

    labels = []
    data_list = []

    if ext == 'csv':
        print(f"Read [{infile}] as CSV file")
        # CSVファイルの処理
        with open(infile, 'r', encoding='utf-8') as csvfile:
            reader = csv.reader(csvfile)
            labels = next(reader)  # 1行目をラベルとして取得
            data_list = [row for row in reader]  # 2行目以降をリストに追加

    elif ext == 'txt':
        print(f"Read [{infile}] as TXT file")
        # テキストファイルの処理（タブ区切り）
        with open(infile, 'r', encoding='utf-8') as txtfile:
            reader = csv.reader(txtfile, delimiter='\t')
            labels = next(reader)  # 1行目をラベルとして取得
            data_list = [row for row in reader]  # 2行目以降をリストに追加

    elif ext == 'xlsx':
        print(f"Read [{infile}] as Excel file")
        # Excelファイルの処理
        workbook = openpyxl.load_workbook(infile, data_only=True)
        sheet = workbook.active  # 最初のシートを取得
        labels = [cell.value for cell in sheet[1]]  # 1行目をラベルとして取得
        for row in sheet.iter_rows(min_row=2, values_only=True):  # 2行目以降をリストに追加
            data_list.append(list(row))

    else:
        raise ValueError("Unsupported file format. Please use .csv, .xlsx, or .txt")

    #data_listの要素をfloat()に変換
    data_list = [[float(value) for value in row] for row in data_list]

    return labels, data_list

def generate_data(cfg, infile = None):
    if infile:
        labels, data_list = read_data(infile)
        x_data = np.array([row[0] for row in data_list])
        y_data = np.array([row[1] for row in data_list])
        cfg.xlabel = labels[0]
        cfg.ylabel = labels[1]  
        true_ai = None
    else:
        x_data = np.linspace(0, 1, cfg.n)
        true_ai = [1.0, -2.0, 3.0]
        for i in range(cfg.p - 3):
            true_ai.append(0.0)
        true_ai = np.array(true_ai)
        noise = cfg.sigma_noise * np.random.randn(cfg.n)
        y_data = model(x_data, true_ai, cfg) + noise

    return x_data, y_data, true_ai

def bayesian_update(x_data, y_data, cfg):
    X = design_matrix(x_data, cfg)
    sigma0_inv = np.linalg.inv(cfg.sigma0)
    SigmaN_inv = sigma0_inv + (1 / cfg.sigma_noise**2) * X.T @ X
    SigmaN = np.linalg.inv(SigmaN_inv)
    muN = SigmaN @ ((1 / cfg.sigma_noise**2) * X.T @ y_data + sigma0_inv @ cfg.mu0)
    return muN, SigmaN

def estimate_noise(x_data, y_data, muN, cfg):
    y_pred = model(x_data, muN, cfg)
    residuals = y_data - y_pred
    return np.std(residuals)

def self_consistent_bayes(x_data, y_data, cfg):
    sigma_old = cfg.sigma_noise
    for i in range(cfg.max_iter):
        muN, SigmaN = bayesian_update(x_data, y_data, cfg)
        sigma_new = estimate_noise(x_data, y_data, muN, cfg)
        rel_diff = abs(sigma_new - sigma_old) / sigma_old

        if rel_diff < cfg.tol:
            print(f"収束: iteration={i+1}, σ_noise={sigma_new:.5f}")
            break
        sigma_old = sigma_new
        cfg.sigma_noise = sigma_new
    else:
        print("警告: σ_noise の自己無撞着計算が収束しませんでした")

    return muN, SigmaN, cfg.sigma_noise

def predictive_mean_and_std(x_data, muN, SigmaN, sigma_noise, cfg):
    X_new = design_matrix(x_data, cfg)
    means = X_new @ muN
    # 信頼区間（ノイズあり）
    variances = np.array([phi @ SigmaN @ phi + sigma_noise**2 for phi in X_new])
    stds = np.sqrt(variances)
    # 信頼区間（ノイズなし）
    variances_wo_noise = np.array([phi @ SigmaN @ phi for phi in X_new])
    stds_wo_noise = np.sqrt(variances_wo_noise)
    return means, stds, stds_wo_noise

def plot_results(x_data, y_data, true_ai, cfg, muN, SigmaN):
    x_plot = np.linspace(min(x_data), max(x_data), 200)
    mean_pred, std_pred, std_wo_noise = predictive_mean_and_std(
        x_plot, muN, SigmaN, cfg.sigma_noise, cfg)
    if true_ai is not None:
        true_y = model(x_plot, true_ai, cfg)
    else:
        true_y = None

    plt.rcParams.update({'font.size': 16})  # フォントサイズ設定
    plt.figure(figsize=(10, 6))
    plt.scatter(x_data, y_data, label="Observed Data", color='blue', alpha=0.5)
    if true_y is not None:
        plt.plot(x_plot, true_y, label="True Function", color='red')
    plt.plot(x_plot, mean_pred, label="Posterior Mean", color='green', linestyle='--')

    # ノイズ込みの信頼区間（予測分布）
    plt.fill_between(
        x_plot,
        mean_pred - std_pred,
        mean_pred + std_pred,
        color='gray',
        alpha=0.3,
        label="1σ Predictive Interval (incl. noise)"
    )

    # モデル不確かさのみ（ノイズ除く）
    plt.fill_between(
        x_plot,
        mean_pred - std_wo_noise,
        mean_pred + std_wo_noise,
        color='orange',
        alpha=0.2,
        label="1σ Model Interval (w/o noise)"
    )

    plt.legend()
    plt.xlabel(cfg.xlabel)
    plt.ylabel(cfg.ylabel)
    plt.title("Bayesian Linear Regression (Self-consistent σ)")
    plt.grid(True)
    plt.show()

def main():
    cfg = initialize()
    update_vars(cfg)
    x_data, y_data, true_ai = generate_data(cfg, cfg.infile)
    muN, SigmaN, sigma_noise = self_consistent_bayes(x_data, y_data, cfg)

    param_std = np.sqrt(np.diag(SigmaN))
    print("\n=== ベイズ線形回帰 結果 ===")
    print("パラメータ推定値（μN）と標準偏差（σN）:")
    for i, (mean, std) in enumerate(zip(muN, param_std)):
        print(f"  a{i}: {mean: .5f} ± {std: .5f}")
    print(f"\n最終的なノイズ標準偏差: σ_noise = {sigma_noise:.5f}")
    print("===========================\n")

    plot_results(x_data, y_data, true_ai, cfg, muN, SigmaN)

if __name__ == "__main__":
    main()
