import sys
import csv
import openpyxl
import numpy as np
import matplotlib.pyplot as plt
from types import SimpleNamespace

def initialize():
    cfg = SimpleNamespace()
    cfg.infile = None
    cfg.xlabel = 'x'
    cfg.ylabel = 'y'
    cfg.p = 5
    cfg.n = 100
    cfg.mu0 = np.zeros(cfg.p)       # 事前平均ベクトル
    cfg.sigma0 = np.eye(cfg.p)      # 事前共分散行列
    cfg.sigma_noise = 1.0           # 初期ノイズ標準偏差
    cfg.em_iters = 100              # エビデンス最適化反復回数
    cfg.em_tol = 1e-4               # 収束判定閾値
    return cfg

def update_vars(cfg):
    for arg in sys.argv[1:]:
        if '=' in arg:
            key, value = arg.split('=',1)
            if hasattr(cfg, key):
                attr = getattr(cfg, key)
                try:
                    if isinstance(attr, bool):
                        setattr(cfg, key, value.lower()=='true')
                    elif isinstance(attr, int):
                        setattr(cfg, key, int(value))
                    elif isinstance(attr, float):
                        setattr(cfg, key, float(value))
                    else:
                        setattr(cfg, key, value)
                except ValueError:
                    print(f"Invalid value for {key}: {value}")
            else:
                print(f"Unknown parameter: {key}")

def basis(i, x):
    return 1.0 if i==0 else x**i

def model(x, a, cfg):
    return sum(a[i]*basis(i,x) for i in range(cfg.p))

def design_matrix(x_data, cfg):
    return np.array([[basis(i,x) for i in range(cfg.p)] for x in x_data])

def read_data(infile):
    ext = infile.split('.')[-1].lower()
    labels, data_list = [], []
    if ext=='csv':
        with open(infile,'r',encoding='utf-8') as f:
            reader=csv.reader(f); labels=next(reader); data_list=list(reader)
    elif ext in ('txt', 'tsv'):
        with open(infile,'r',encoding='utf-8') as f:
            reader=csv.reader(f, delimiter='\t'); labels=next(reader); data_list=list(reader)
    elif ext=='xlsx':
        wb=openpyxl.load_workbook(infile,data_only=True)
        sheet=wb.active; labels=[c.value for c in sheet[1]]
        for row in sheet.iter_rows(min_row=2, values_only=True): data_list.append(list(row))
    else:
        raise ValueError("Unsupported format")
    data = [[float(v) for v in row] for row in data_list]
    return labels, data

def generate_data(cfg):
    if cfg.infile:
        labels, data = read_data(cfg.infile)
        x=np.array([r[0] for r in data]); y=np.array([r[1] for r in data])
        cfg.xlabel, cfg.ylabel = labels[0], labels[1]
        return x,y
    x=np.linspace(0,1,cfg.n)
    true_a=np.zeros(cfg.p); true_a[:3]=[1.0,-2.0,3.0]
    noise=cfg.sigma_noise*np.random.randn(cfg.n)
    y=np.array([model(xi,true_a,cfg) for xi in x])+noise
    return x,y

def evidence_maximization(x, y, cfg):
    N=len(x); M=cfg.p
    Phi=design_matrix(x,cfg)
    sigma0_inv=np.linalg.inv(cfg.sigma0)
    alpha=1.0  # initial prior precision
    beta=1.0/(cfg.sigma_noise**2)
    eigvals=np.linalg.eigvalsh(Phi.T@Phi)
    m0=cfg.mu0
    I=np.eye(M)
    for _ in range(cfg.em_iters):
        S_inv=alpha*sigma0_inv+beta*(Phi.T@Phi)
        S=np.linalg.inv(S_inv)
        m=beta*(S@Phi.T@y)+alpha*(S@sigma0_inv@m0)
        gamma=np.sum(beta*eigvals/(alpha+beta*eigvals))
        alpha_new=gamma/((m-m0)@ (m-m0))
        res=y-Phi@m
        beta_new=(N-gamma)/(res@res)
        if abs(alpha_new-alpha)<cfg.em_tol*alpha and abs(beta_new-beta)<cfg.em_tol*beta:
            alpha,beta=alpha_new,beta_new; break
        alpha, beta=alpha_new,beta_new
    S_inv=alpha*sigma0_inv+beta*(Phi.T@Phi)
    S=np.linalg.inv(S_inv)
    m=beta*(S@Phi.T@y)+alpha*(S@sigma0_inv@m0)
    sigma_noise=np.sqrt(1.0/beta)
    return m, S, sigma_noise, alpha, beta

def predictive(x, m, S, sigma_noise, cfg):
    Phi_new=design_matrix(x,cfg)
    mean=Phi_new@m
    var_model=np.sum((Phi_new@S)*Phi_new,axis=1)
    std_model=np.sqrt(var_model)
    std_pred=np.sqrt(var_model+sigma_noise**2)
    return mean, std_pred, std_model

def plot_results(x, y, cfg, m, S, sigma_noise):
    xs=np.linspace(min(x),max(x),200)
    mean, std_pred, std_model=predictive(xs,m,S,sigma_noise,cfg)
    plt.figure(figsize=(8,6))
    plt.scatter(x,y,label='Data',color='blue',alpha=0.5)
    plt.plot(xs,mean,label='Mean',color='green')
    plt.fill_between(xs,mean-std_model,mean+std_model,alpha=0.3,label='±1σ model')
    plt.fill_between(xs,mean-std_pred,mean+std_pred,alpha=0.2,label='±1σ pred')
    plt.xlabel(cfg.xlabel); plt.ylabel(cfg.ylabel)
    plt.title(f'Bayesian Regression (p={cfg.p}), σ_noise={sigma_noise:.3f}')
    plt.legend(); plt.grid(True); plt.show()

def main():
    cfg=initialize(); update_vars(cfg)
    x,y=generate_data(cfg)
    m, S, sigma_noise, alpha, beta=evidence_maximization(x,y,cfg)
    stds=np.sqrt(np.diag(S))
    print(f"Estimated σ_noise={sigma_noise:.4f}, α={alpha:.4f}, β={beta:.4f}")
    for i,(mi,si) in enumerate(zip(m,stds)):
        print(f"a{i}: {mi:.4f} ±{si:.4f}")
    plot_results(x,y,cfg,m,S,sigma_noise)

if __name__=='__main__': main()
